package position

/*
 * Copyright (c) 2022. Henrik Bærbak Christensen, Aarhus University.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Package containing Position and Direction aspects of SkyCave

import (
	"regexp"
	"strconv"
)

// Record of a point in 3 D space
type Point3 struct {
	X, Y, Z int
}

// Define the string format for a Point3 = "(x,y,z)" where
// x,y,z can be a signed integer
var re = regexp.MustCompile(`\((-?\d+),(-?\d+),(-?\d+)\)`)

// Given a string representation, return the Point3
func IsValidPositionString(positionString string) bool {
	match := re.MatchString(positionString)
	return match
}

// Given a string representation, return the Point3
// PRECOND: must be a correct string, see IsValidPositionString
func ParseString(positionString string) Point3 {
	parts := re.FindStringSubmatch(positionString)
	x, _ := strconv.Atoi(parts[1]) 
	y, _ := strconv.Atoi(parts[2]) 
	z, _ := strconv.Atoi(parts[3]) 

	return Point3{x,y,z}
}

// public enum Direction
type Direction int
const (
	NORTH Direction = iota
	SOUTH
	EAST 
	WEST 
	UP 
	DOWN 
)
// toString() function for directions
func (d Direction) String() string {
	switch d {
	case NORTH:
		return "NORTH"
	case SOUTH:
		return "SOUTH"
	case EAST:
		return "EAST"
	case WEST:
		return "WEST"
	case UP:
		return "UP"
	case DOWN:
		return "DOWN"
	}
	return "unknown"
}

// Direction offset table to compute (p,dir) -> (new_p)
var offset = [3][6]int{
	{ 0, 0, +1, -1, 0, 0 },
	{ 1, -1, 0, 0, 0, 0 },
	{ 0, 0, 0, 0, 1, -1 }}

// Given a P and Dir, return a new P in that direction
func Translate(p Point3, dir Direction) Point3 {
	newP := Point3{
		p.X + offset[0][dir],
		p.Y + offset[1][dir],
		p.Z + offset[2][dir]}
	return newP
}
