package main

/*
 * Copyright (c) 2022. Henrik Bærbak Christensen, Aarhus University.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// March 2022 Kata: Rewrite CaveService in Go as a learning experience.

// To run: Install go using e.g 'sudo snap install go --classic'
// then 'go run server.go' to run the cave server on localhost:9999

// OR - build the container using the Dockerfile

import (
	"fmt"
	store "caveservice/storage"
	p3 "caveservice/position"
	"github.com/gin-gonic/gin"
	"net/http"
	"encoding/json"
)

var router *gin.Engine

func main() {
	fmt.Println("--- Sitting at a Car repair shop, waiting for Tire change ---")
	// Initialize the DB
	store.PopulateDatabase();

	// Initialize the web server on port 9999
	// gin.SetMode(gin.ReleaseMode)
	// router = gin.New()

	router = gin.Default()
	initializeRoutes()
	router.Run(":9999")
}

func initializeRoutes() {
	// GET /msdo/v1/cave/room/{positionString}
	router.GET("/msdo/v1/cave/room/:positionString", handleGetRoom)
	
	// GET /msdo/v1/cave/room/{positionString}/exits
	router.GET("/msdo/v1/cave/room/:positionString/exits",
		handleGetExits)

	// POST /msdo/v1/cave/room/{positionString}
	router.POST("/msdo/v1/cave/room/:positionString",
		handlePostRoom)
}

var nullObject = []byte("{}")

func handleGetRoom(c *gin.Context) {
	posString := c.Param("positionString")
	// Validate bad request
	if ! p3.IsValidPositionString(posString) {
		c.Data(http.StatusBadRequest,
			"application/json",
			nullObject)
		return
	}

	newRecord, exists := store.GetRoom(posString)

	// NOT FOUND in case there is not such room
	if ! exists {
		c.Data(http.StatusNotFound,
			"application/json",
			nullObject)
		return
	}

	// Otherwise, return JSON version
        rJson, err := json.Marshal(newRecord)
        if err != nil {
		panic(err)
        }


	c.Data(http.StatusOK, "application/json", rJson)	
}

func handleGetExits(c *gin.Context) {
	posString := c.Param("positionString")

	// Validate bad request
	if ! p3.IsValidPositionString(posString) {
		c.Data(http.StatusBadRequest,
			"application/json",
			nullObject)
		return
	}

	_, exists := store.GetRoom(posString)
	// NOT FOUND in case there is not such room
	if ! exists {
		c.Data(http.StatusNotFound, "application/json",
			nullObject)
		return
	}

	exits := store.GetExits(posString)

	// Otherwise, return JSON version
        rJson, err := json.Marshal(exits)
        if err != nil {
		panic(err)
        }


	c.Data(http.StatusOK, "application/json", rJson)	
}

func handlePostRoom(c *gin.Context) {
	posString := c.Param("positionString")

	// Validate bad request
	if ! p3.IsValidPositionString(posString) {
		c.Data(http.StatusBadRequest,
			"application/json",
			nullObject)
		return
	}

	var roomRecord store.RoomRecord

	err := c.ShouldBindJSON(&roomRecord)

	if err != nil {
		panic(err)
	}

	// Try to store the record
	statusCode := store.UpsertRoom(posString, roomRecord)
	if statusCode != http.StatusCreated &&
		statusCode != http.StatusOK {
		c.Data(statusCode, "application/json", nullObject)
		return
	}
	// Bit cumbersome but have to fetch the record to
	// get all values
	roomRecord, _ = store.GetRoom(posString)

	// And set the header 'Location'
	c.Header("Location", c.Request.Host+c.Request.URL.Path)
	c.JSON(statusCode, roomRecord)
}
