package storage

/*
 * Copyright (c) 2022. Henrik Bærbak Christensen, Aarhus University.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// FakeObject Storage implementation of RoomRecords for CaveService

import (
	"net/http" // For HTTP status codes
	"github.com/google/uuid" // UUID
	"time" // for ISO 8601
	"strconv"
	p3 "caveservice/position"
)

// The central room record
type RoomRecord struct {
    Id                     string `json:"id"`
    CreationTimeISO8601    string `json:"creationTimeISO8601"`
    Description            string `json:"description"`
    CreatorId              string `json:"creatorId"`
}

// Define the database
var database = make(map[string]RoomRecord)

// The special ID of the original rooms in the cave
var WILL_CROWTHER = "0"

// Populate database
func PopulateDatabase() {
	UpsertRoom( "(0,0,0)",
		RoomRecord{
			Id: "na",
			CreationTimeISO8601: "na",
			Description: "You are standing at the end of a road before a small brick building.", 
			CreatorId: WILL_CROWTHER})
	
	UpsertRoom( "(0,1,0)",
		RoomRecord{
			Id: "27aaa69d-0d14-4f2f-b57d-e62c731170ed",
			CreationTimeISO8601: "2022-03-07T08:53:29.205879+01:00",
			Description: "You are in open forest, with a deep valley to one side.", 
			CreatorId: WILL_CROWTHER })

	UpsertRoom( "(1,0,0)",
		RoomRecord{
			Id: "37aaa69d-0d14-4f2f-b57d-e62c731170ed",
			CreationTimeISO8601: "2022-03-07T08:53:29.205879+01:00",
			Description: "You are inside a building, a well house for a large spring.", 
			CreatorId: WILL_CROWTHER })

	UpsertRoom( "(-1,0,0)",
		RoomRecord{
			Id: "47aaa69d-0d14-4f2f-b57d-e62c731170ed",
			CreationTimeISO8601: "2022-03-07T08:53:29.205879+01:00",
			Description: "You have walked up a hill, still in the forest.", 
			CreatorId: WILL_CROWTHER })

	UpsertRoom( "(0,0,1)",
		RoomRecord{
			Id: "57aaa69d-0d14-4f2f-b57d-e62c731170ed",
			CreationTimeISO8601: "2022-03-07T08:53:29.205879+01:00",
			Description: "You are in the top of a tall tree, at the end of a road.", 
			CreatorId: WILL_CROWTHER })
}

func GetRoom(positionString string) (RoomRecord, bool) {
	currentRecord, exists := database[positionString]
	return currentRecord, exists
}

func UpsertRoom(positionString string, record RoomRecord) int {
	statusCode := http.StatusCreated
	currentRecord, exists := GetRoom(positionString)
	if exists {
		// Room exists, which may signal an update
		// which is only allowed if creator is same
		// as author
		if currentRecord.CreatorId == record.CreatorId {
			statusCode = http.StatusOK
		} else {
			return http.StatusUnauthorized
		}
	}
	
	// Simulate classic DB behaviour: timestamp record and
	// assign unique id

	// Overwrite time stamp with current time in ISO 8601
	record.CreationTimeISO8601 = time.Now().Format(time.RFC3339)

	// Create UUID
	record.Id = uuid.New().String()

	// And insert into DB
	database[positionString] = record
	return statusCode
}


func GetExits(positionString string) []string {
	var exits [] string
	var origo = p3.ParseString(positionString)
	// Iterate the 6 directions out from origo
	for dir := p3.NORTH; dir <= p3.DOWN; dir++ {
		// find the point in the cave when moving this direction
		p := p3.Translate(origo, dir)
		// Convert to string as used in db
		pString := "(" + strconv.Itoa(p.X) + "," +
			strconv.Itoa(p.Y) + "," + strconv.Itoa(p.Z) + ")"

		// verify if map has this room
		_, ok := database[pString]
		
		if ok {
			// If so, convert enum to string
			s := dir.String()
			// and append to exit list
			exits = append(exits, s)
		}
	}
	return exits
	
}	

